#!/bin/bash
# This script reads and prints the FPGA audio input registers.

TOOLS=/usr/local/bin

# Check SV2 driver loaded.
lsmod | grep sv2 > /dev/null
if [ "$?" == "0" ]; then
    I2CTOOL=$TOOLS/MtxI2cTool
    BOARDINFO=$TOOLS/MtxBoardInfo
else
    echo "SV2 driver is not loaded."
    exit 1
fi

function PrintAinStatus
{
    echo "Audio input $1"
    echo "{"
    ioaddr=$2
    bus=$3
    addr=$4

    ctrl0=$($I2CTOOL               -b $bus -a $addr -o16 $((ioaddr + 0x8))  -r 1 | awk '{ print $2 }')
    ctrl1=$($I2CTOOL               -b $bus -a $addr -o16 $((ioaddr + 0x9))  -r 1 | awk '{ print $2 }')
    bufsize=$($I2CTOOL             -b $bus -a $addr -o16 $((ioaddr + 0xC))  -r 1 | awk '{ print $2 }')
    status=$($I2CTOOL              -b $bus -a $addr -o16 $((ioaddr + 0x10)) -r 1 | awk '{ print $2 }')
    #xilstatus=$($I2CTOOL           -b $bus -a $addr -o16 $((ioaddr + 0x11)) -r 1 | awk '{ print $2 }')
    nbtick0=$($I2CTOOL             -b $bus -a $addr -o16 $((ioaddr + 0x18)) -r 1 | awk '{ print $2 }' | sed 's/0x//g')
    nbtick1=$($I2CTOOL             -b $bus -a $addr -o16 $((ioaddr + 0x19)) -r 1 | awk '{ print $2 }' | sed 's/0x//g')
    nbtick2=$($I2CTOOL             -b $bus -a $addr -o16 $((ioaddr + 0x1A)) -r 1 | awk '{ print $2 }' | sed 's/0x//g')
    nbtick3=$($I2CTOOL             -b $bus -a $addr -o16 $((ioaddr + 0x1B)) -r 1 | awk '{ print $2 }' | sed 's/0x//g')
    audioclkfrq_nbvalid=$($I2CTOOL -b $bus -a $addr -o16 $((ioaddr + 0x1C)) -r 1 | awk '{ print $2 }')
    parity=$($I2CTOOL              -b $bus -a $addr -o16 $((ioaddr + 0x1D)) -r 1 | awk '{ print $2 }')
    channelstatus=$($I2CTOOL       -b $bus -a $addr -o16 $((ioaddr + 0x20)) -r 5 | awk '{ print $2 }' | tr -s '\n' ' ' | sed 's/0x//g')
    iec60937_status0=$($I2CTOOL    -b $bus -a $addr -o16 $((ioaddr + 0x30)) -r 1 | awk '{ print $2 }')
    iec60937_burstinfo=$($I2CTOOL  -b $bus -a $addr -o16 $((ioaddr + 0x34)) -r 1 | awk '{ print $2 }')

    nbtick=$(echo "$nbtick3$nbtick2$nbtick1$nbtick0")
    nbtick_dec=$(echo "$((16#$nbtick))")
    nbtick_dec=$((((nbtick_dec*48000000)/8388608)/128))

    parity_errcnt=$(((parity & 0xFC) >> 2))
    parity_errcnt=$(echo "$((16#$parity_errcnt))")

    # Get protocol string
    ctrl0_mode=$(((ctrl0 & 0x30) >> 4))
    case $ctrl0_mode in
    0)
        protocol="SPDIF"
        ;;
    1) 
        protocol="I2S"
        ;;
    *)
        protocol="UNKNOWN"
        ;;
    esac

    # Get DATAMODE string
    ctrl1_datamode=$((ctrl1 & 0x07))
    case $ctrl1_datamode in  
    0) 
        datamode="Disabled"
        ;;
    1)
        datamode="16 bits PCM"
        ;;
    2)
        datamode="24 bits PCM"
        ;;
    3) 
        datamode="Raw data mode"
        ;;
    4)
        datamode="Non-linear PCM"
        ;;
    *)
        datamode="UNKNOWN"
        ;;
    esac

    # Get IEC60937 DATATYPE string
    iec60937_burstinfo_datatype=$((iec60937_burstinfo & 0x1F))
    case $iec60937_burstinfo_datatype in 
    0)
        iec60937_datatype="Null Data"
        ;;
    1)
        iec60937_datatype="AC3 Data"
        ;;
    2)
        iec60937_datatype="Refer to SMPTE 338M"
        ;;
    3)
        iec60937_datatype="Pause"
        ;;
    4)
        iec60937_datatype="MPEG-1 layer-1 data"
        ;;
    5)
        iec60937_datatype="MPEG-1 layer-2 or -3 data or MPEG-2 without extension"
        ;;
    6)
        iec60937_datatype="MPEG-2 data with extension"
        ;;
    7)
        iec60937_datatype="MPEG-2 AAC"
        ;;
    8)
        iec60937_datatype="MPEG-2, layer-1 low sampling frequency"
        ;;
    9)
        iec60937_datatype="MPEG-2, layer-2 low sampling frequency"
        ;;
    10)
        iec60937_datatype="MPEG-2, layer-3 low sampling frequency"
        ;;
    11)
        iec60937_datatype="DTS type I"
        ;;
    12)
        iec60937_datatype="DTS type II"
        ;;
    13)
        iec60937_datatype="DTS type III"
        ;;
    14)
        iec60937_datatype="ATRAC"
        ;;
    15)
        iec60937_datatype="ATRAC 2/3"
        ;;
    *)
        iec60937_datatype="UNKNOWN"
        ;;
    esac

    echo "  CTRL0..................$ctrl0"
    echo "   .MODE...................$protocol"
    echo "   .RESET..................$(((ctrl0 & 0x02) != 0))"
    echo "   .ENABLE.................$(((ctrl0 & 0x01) != 0))"
    echo "  CTRL1.DATAMODE.........$datamode"
    echo "  BUFFERSIZE.............$bufsize ($(((bufsize+1)*256)))"
    echo "  STATUS.................$status"
    echo "   .CHST_CHNG..............$(((status & 0x20) != 0))"
    echo "   .CHST_VALID.............$(((status & 0x10) != 0))"
    echo "   .CHST_VALID_FALL........$(((status & 0x08) != 0))"
    echo "   .CHST_VALID_RISE........$(((status & 0x04) != 0))"
    echo "   .SIGNAL_LOCKED..........$(((status & 0x02) != 0))"
    #echo "  XIL_STATUS.............$xilstatus"
    echo "  CLKFRQ_NBTICK..........0x$nbtick ($nbtick_dec)"
    echo "  CLKFRQ_NBVALID.........$audioclkfrq_nbvalid"
    echo "  PARITY.................$parity"
    echo "   .ERROR_CNT..............$parity_errcnt"
    echo "   .PARITY_OK..............$(((parity & 0x01) != 0))"
    echo "  CHANNEL_STATUS.........$channelstatus"
    echo "  IEC60937_STATUS0.......$iec60937_status0"
    echo "   .LPCM_TO_NLPCM_DET......$(((iec60937_status0 & 0x04) != 0))"
    echo "   .NLPCM_TO_LPCM_DET......$(((iec60937_status0 & 0x02) != 0))"
    echo "   .DATABURST_DET..........$(((iec60937_status0 & 0x01) != 0))"
    echo "  IEC60937_BURSTINFO.....$iec60937_burstinfo"
    echo "   .DATATYPE...............$iec60937_datatype"

    echo "}"
    echo ""
}

ain_index=-1

# Check arguments.
if [ ! -z $1 ]; then
    if [ "$1" == "--help" ] || [ "$1" == "-h" ]; then
        echo "Reads and prints the FPGA audio input registers. If ain_index is inspecified, the status of every input is printed."
        echo "Usage:"
        echo "  $0 [ain_index]"
        exit 0
    elif [ $1 -eq $1 ] && [ $1 -ge 0 ] 2>/dev/null; then
        ain_index=$1
    else
        echo "Invalid parameter $1"
        exit 1
    fi
fi

boardId=$($BOARDINFO | grep ProductName | cut -f20 -d' ')

# Extio3 Quad TX
if [ "$boardId" == "XTO3-N3408-ATX" ] || [ "$boardId" == "XTO3-N3408-ETX" ]; then
    echo "Extio3 Quad TX"
    if [ "$ain_index" == "-1" ]; then    
        PrintAinStatus 0 0x4180 1 0xD4
        PrintAinStatus 1 0x4180 1 0xD2
        PrintAinStatus 2 0x4180 0 0xD4
        PrintAinStatus 3 0x4180 0 0xD2
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4180 1 0xD4        
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4180 1 0xD2        
    elif [ "$ain_index" == "2" ]; then
        PrintAinStatus 2 0x4180 0 0xD4
    elif [ "$ain_index" == "3" ]; then
        PrintAinStatus 3 0x4180 0 0xD2
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
# Extio3 Dual TX
elif [ "$boardId" == "XTO3-N3208-ATX" ] || [ "$boardId" == "XTO3-N3208-ETX" ]; then
    echo "Extio3 Dual TX"  
    if [ "$ain_index" == "-1" ]; then
        PrintAinStatus 0 0x4180 0xD4
        PrintAinStatus 1 0x4180 0xD2
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4180 0xD4        
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4180 0xD2
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
# Maevex6150 Quad
elif [ "$boardId" == "MAEVEX-6150-QUAD" ] || [ "$boardId" == "MAEVEX-6150" ] || [ "$boardId" == "MAEVEX-6150 V0" ]; then
    echo "Maevex6150 Quad"
    if [ "$ain_index" == "-1" ]; then
        PrintAinStatus 0 0x4100 1 0xD4
        PrintAinStatus 1 0x4100 1 0xD2
        PrintAinStatus 2 0x4100 0 0xD4
        PrintAinStatus 3 0x4100 0 0xD2
        PrintAinStatus 4 0x4180 1 0xD4
        PrintAinStatus 5 0x4180 1 0xD2
        PrintAinStatus 6 0x4180 0 0xD4
        PrintAinStatus 7 0x4180 0 0xD2
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4100 1 0xD4        
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4100 1 0xD2        
    elif [ "$ain_index" == "2" ]; then
        PrintAinStatus 2 0x4100 0 0xD4
    elif [ "$ain_index" == "3" ]; then
        PrintAinStatus 3 0x4100 0 0xD2
    elif [ "$ain_index" == "4" ]; then
        PrintAinStatus 4 0x4180 1 0xD4        
    elif [ "$ain_index" == "5" ]; then
        PrintAinStatus 5 0x4180 1 0xD2        
    elif [ "$ain_index" == "6" ]; then
        PrintAinStatus 6 0x4180 0 0xD4
    elif [ "$ain_index" == "7" ]; then
        PrintAinStatus 7 0x4180 0 0xD2
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
# Maevex6100 Quad
elif [ "$boardId" == "MAEVEX2" ]; then
    echo "Maevex6100 Quad"
    if [ "$ain_index" == "-1" ]; then
        PrintAinStatus 0 0x4100 0 0xD2
        PrintAinStatus 1 0x4100 0 0xD4
        PrintAinStatus 2 0x4100 1 0xD2
        PrintAinStatus 3 0x4100 1 0xD4
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4100 0 0xD2
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4100 0 0xD4
    elif [ "$ain_index" == "2" ]; then
        PrintAinStatus 2 0x4100 1 0xD2
    elif [ "$ain_index" == "3" ]; then
        PrintAinStatus 3 0x4100 1 0xD4
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
# Mura IPX Dual DP1.2
elif [ "$boardId" == "MURA-IPX-I2UEF" ]; then
    echo "Mura IPX Dual DP1.2"
    if [ "$ain_index" == "-1" ]; then
        PrintAinStatus 0 0x4180 0 0xD2
        PrintAinStatus 1 0x4180 0 0xD6
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4180 0 0xD2
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4180 0 0xD6
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
# EVM1
elif [ "$boardId" == "EVM1" ]; then
    echo "EVM1"
    if [ "$ain_index" == "-1" ]; then
        PrintAinStatus 0 0x4180 0 0xD2
        PrintAinStatus 1 0x4200 0 0xD2
        PrintAinStatus 2 0x4180 0 0xD4
        PrintAinStatus 3 0x4200 0 0xD4
    elif [ "$ain_index" == "0" ]; then
        PrintAinStatus 0 0x4180 0 0xD2
    elif [ "$ain_index" == "1" ]; then
        PrintAinStatus 1 0x4200 0 0xD2
    elif [ "$ain_index" == "2" ]; then
        PrintAinStatus 2 0x4180 0 0xD4
    elif [ "$ain_index" == "3" ]; then
        PrintAinStatus 3 0x4200 0 0xD4
    else
        echo "Invalid audio input index $ain_index"
        exit 1
    fi
else
    echo "Unsupported board ($boardId)"
    exit 1
fi

exit 0

